from dataclasses import dataclass
from typing import Optional, Tuple


@dataclass
class TrainCfg:
    # general task params
    task: int = 0
    cost_limit: float = 1
    device: str = "cuda:6"
    thread: int = 4  # if use "cpu" to train
    seed: int = 0
    # algorithm params
    actor_lr: float = 1e-4
    critic_lr: float = 3e-4
    hidden_sizes: Tuple[int, ...] = (128, 128, 128)
    auto_alpha: bool = True
    alpha_lr: float = 1e-4
    target_entropy_ratio: float = 0.98
    alpha: float = 0.005
    tau: float = 0.05
    n_step: int = 1
    conditioned_sigma: bool = True
    unbounded: bool = False
    last_layer_scale: bool = False
    # Lagrangian specific arguments
    use_lagrangian: bool = True
    use_shield: bool = False
    llm_shield: bool = False
    use_fake_data: bool = False
    fake_next_obs: bool = False
    fake_done: bool = False
    int_reward: bool = False
    fake_rew: float = -1.0
    fake_cost: float = 10.0
    fake_weight: float = 1.0
    shield_prop: float = 1.0
    lagrangian_pid: Tuple[float, ...] = (0.05, 0.0005, 0.1)
    lagrangian_max: float = 100.0
    rescaling: bool = True
    # Base policy common arguments
    gamma: float = 0.97
    deterministic_eval: bool = False
    action_scaling: bool = False
    max_cost_q: bool = False
    action_bound_method: str = "clip"
    # collecting params
    epoch: int = 100
    episode_per_collect: int = 2
    step_per_epoch: int = 1000
    update_per_step: float = 0.4
    buffer_size: int = 200000
    worker: str = "ShmemVectorEnv"
    training_num: int = 2
    testing_num: int = 20
    # general train params
    batch_size: int = 4096
    reward_threshold: float = 10000  # for early stop purpose
    save_interval: int = 4
    resume: bool = False  # TODO
    save_ckpt: bool = True  # set this to True to save the policy model
    verbose: bool = True
    render: bool = False
    # logger params
    logdir: str = "logs"
    project: str = "harzard_world"
    group: Optional[str] = None
    name: Optional[str] = None
    prefix: Optional[str] = "sacl_disc"
    suffix: Optional[str] = ""
    use_default_cfg: bool = False


# bullet-safety-gym task default configs


@dataclass
class Bullet1MCfg(TrainCfg):
    epoch: int = 100


@dataclass
class Bullet5MCfg(TrainCfg):
    epoch: int = 500


@dataclass
class Bullet10MCfg(TrainCfg):
    epoch: int = 1000


# safety gymnasium task default configs


@dataclass
class MujocoBaseCfg(TrainCfg):
    task: str = "SafetyPointCircle1Gymnasium-v0"
    epoch: int = 250
    cost_limit: float = 25
    gamma: float = 0.99
    n_step: int = 3
    # collecting params
    step_per_epoch: int = 20000
    episode_per_collect = 5
    buffer_size: int = 800000


@dataclass
class Mujoco2MCfg(MujocoBaseCfg):
    epoch: int = 100


@dataclass
class Mujoco20MCfg(MujocoBaseCfg):
    epoch: int = 1000


@dataclass
class Mujoco10MCfg(MujocoBaseCfg):
    epoch: int = 500
